/**
 * Discord Webhook Logger
 * Logs all important events to Discord
 */

const WEBHOOK_URL = process.env.DISCORD_WEBHOOK_URL

type LogType = 'info' | 'success' | 'warning' | 'error' | 'creator' | 'studio' | 'payment' | 'submission'

const COLORS: Record<LogType, number> = {
    info: 0x3498db,
    success: 0x2ecc71,
    warning: 0xf39c12,
    error: 0xe74c3c,
    creator: 0x9b59b6,
    studio: 0x1abc9c,
    payment: 0x2ecc71,
    submission: 0xe67e22,
}

const EMOJIS: Record<LogType, string> = {
    info: 'ℹ️',
    success: '✅',
    warning: '⚠️',
    error: '❌',
    creator: '🎮',
    studio: '🏢',
    payment: '💰',
    submission: '📤',
}

export interface DiscordField {
    name: string
    value: string
    inline?: boolean
}

export async function logToDiscord(
    title: string,
    description: string,
    type: LogType = 'info',
    fields?: DiscordField[]
) {
    if (!WEBHOOK_URL) {
        console.warn('[Discord] No webhook URL configured')
        return
    }

    const embed = {
        title: `${EMOJIS[type]} ${title}`,
        description,
        color: COLORS[type],
        fields: fields?.map(f => ({
            name: f.name,
            value: f.value.substring(0, 1024), // Discord limit
            inline: f.inline ?? false
        })),
        timestamp: new Date().toISOString(),
        footer: {
            text: 'Swiplay V2'
        }
    }

    try {
        await fetch(WEBHOOK_URL, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ embeds: [embed] })
        })
    } catch (error) {
        console.error('[Discord] Failed to send log:', error)
    }
}

// Convenience methods
export const discord = {
    info: (title: string, desc: string, fields?: DiscordField[]) => logToDiscord(title, desc, 'info', fields),
    success: (title: string, desc: string, fields?: DiscordField[]) => logToDiscord(title, desc, 'success', fields),
    warning: (title: string, desc: string, fields?: DiscordField[]) => logToDiscord(title, desc, 'warning', fields),
    error: (title: string, desc: string, fields?: DiscordField[]) => logToDiscord(title, desc, 'error', fields),
    creator: (title: string, desc: string, fields?: DiscordField[]) => logToDiscord(title, desc, 'creator', fields),
    studio: (title: string, desc: string, fields?: DiscordField[]) => logToDiscord(title, desc, 'studio', fields),
    payment: (title: string, desc: string, fields?: DiscordField[]) => logToDiscord(title, desc, 'payment', fields),
    submission: (title: string, desc: string, fields?: DiscordField[]) => logToDiscord(title, desc, 'submission', fields),
}
