import nodemailer from 'nodemailer'
import { getTranslations } from 'next-intl/server'

const transporter = nodemailer.createTransport({
  host: process.env.SMTP_HOST || 'smtp.gmail.com',
  port: parseInt(process.env.SMTP_PORT || '587'),
  secure: false,
  auth: {
    user: process.env.SMTP_USER,
    pass: process.env.SMTP_PASS,
  },
})

const styles = {
  body: 'font-family: "Inter", -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif; line-height: 1.6; color: #e0e0e0; background-color: #000000; margin: 0; padding: 0;',
  container: 'max-width: 600px; margin: 0 auto; padding: 40px 20px; background-color: #000000;',
  header: 'text-align: center; margin-bottom: 40px; padding: 20px 0;',
  logoContainer: 'display: inline-flex; align-items: center; justify-content: center; gap: 4px;',
  logoText: 'font-size: 32px; font-weight: 800; color: #ffffff; letter-spacing: -1px; text-decoration: none; line-height: 1;',
  logoImage: 'height: 32px; width: auto; vertical-align: middle; margin: 0 2px;',
  content: 'background: #09090b; border: 1px solid #27272a; border-radius: 24px; padding: 40px; box-shadow: 0 4px 24px rgba(0,0,0,0.5);',
  h1: 'color: #ffffff; font-size: 24px; font-weight: 700; margin-bottom: 24px; text-align: center; letter-spacing: -0.5px;',
  text: 'color: #a1a1aa; font-size: 16px; margin-bottom: 24px; text-align: center; line-height: 1.6;',
  codeBox: 'background: linear-gradient(135deg, rgba(147, 51, 234, 0.1) 0%, rgba(124, 58, 237, 0.1) 100%); border: 1px solid rgba(147, 51, 234, 0.2); color: white; padding: 32px 20px; border-radius: 16px; text-align: center; margin: 32px 0; box-shadow: 0 0 40px rgba(147, 51, 234, 0.1);',
  code: 'font-size: 32px; font-weight: 800; letter-spacing: 4px; margin: 0; font-family: monospace; color: #ffffff; text-shadow: 0 0 20px rgba(147, 51, 234, 0.5);',
  codeLabel: 'font-size: 11px; text-transform: uppercase; letter-spacing: 3px; color: #a1a1aa; margin-bottom: 16px; font-weight: 600;',
  footer: 'text-align: center; color: #52525b; font-size: 12px; margin-top: 40px; border-top: 1px solid #27272a; padding-top: 20px;',
  button: 'display: inline-block; background: #ffffff; color: #000000; padding: 16px 32px; border-radius: 12px; font-weight: 600; text-decoration: none; margin-top: 20px; transition: all 0.2s;',
  infoBox: 'background: #18181b; border: 1px solid #27272a; border-radius: 12px; padding: 20px; margin: 20px 0;',
  infoRow: 'display: flex; justify-content: space-between; padding: 12px 0; border-bottom: 1px solid #27272a;',
  infoLabel: 'color: #71717a; font-weight: 500;',
  infoValue: 'color: #ffffff; font-weight: 600;'
}

const LOGO_URL = 'https://swiplay.com/swiplay-logo.png'

export async function sendLoginCode(email: string, code: string, role: 'studio' | 'creator' | 'admin', locale: string = 'fr') {
  const t = await getTranslations({ locale, namespace: 'Emails.Login' })
  const roleText = role === 'studio' ? 'Studio' : role === 'admin' ? 'Admin' : t('creator_role')

  await transporter.sendMail({
    from: `"Swiplay" <${process.env.SMTP_USER}>`,
    to: email,
    subject: t('subject'),
    html: `
      <!DOCTYPE html>
      <html>
        <head>
          <meta charset="utf-8">
          <title>${t('title')}</title>
        </head>
        <body style="${styles.body}">
          <div style="${styles.container}">
            <div style="${styles.header}">
              <div style="${styles.logoContainer}">
                <span style="${styles.logoText}">SWIPLAY</span>
              </div>
            </div>
            
            <div style="${styles.content}">
              <h1 style="${styles.h1}">${t('heading', { role: roleText })}</h1>
              <p style="${styles.text}">${t('instruction')}</p>
              
              <div style="${styles.codeBox}">
                <div style="${styles.codeLabel}">${t('code_label')}</div>
                <div style="${styles.code}">${code}</div>
              </div>
              
              <p style="${styles.text}; font-size: 14px; color: #71717a;">
                ${t('validity')}<br>
                ${t('warning')}
              </p>
            </div>
            
            <div style="${styles.footer}">
              <p>© 2025 Swiplay. ${t('rights')}</p>
            </div>
          </div>
        </body>
      </html>
    `,
  })
}

export async function sendAccessRequestNotification(email: string, company: string, locale: string = 'fr') {
  const t = await getTranslations({ locale, namespace: 'Emails.AccessRequest' })

  await transporter.sendMail({
    from: `"Swiplay" <${process.env.SMTP_USER}>`,
    to: process.env.ADMIN_EMAIL || 'contact@swiplay.com',
    subject: `🔥 Nouvelle demande Studio - ${company}`,
    html: `
      <!DOCTYPE html>
      <html>
        <head>
          <meta charset="utf-8">
        </head>
        <body style="${styles.body}">
          <div style="${styles.container}">
            <div style="${styles.header}">
              <div style="${styles.logoContainer}">
                <span style="${styles.logoText}">SWIPLAY</span>
              </div>
            </div>
            
            <div style="${styles.content}; border-color: #ef4444;">
              <h1 style="${styles.h1}">Nouvelle demande d'accès</h1>
              
              <div style="${styles.infoBox}">
                <div style="${styles.infoRow}">
                  <span style="${styles.infoLabel}">Entreprise</span>
                  <span style="${styles.infoValue}">${company}</span>
                </div>
                <div style="${styles.infoRow}">
                  <span style="${styles.infoLabel}">Email</span>
                  <span style="${styles.infoValue}">${email}</span>
                </div>
                <div style="${styles.infoRow}; border-bottom: none;">
                  <span style="${styles.infoLabel}">Date</span>
                  <span style="${styles.infoValue}">${new Date().toLocaleDateString('fr-FR')}</span>
                </div>
              </div>
              
              <div style="text-align: center;">
                <a href="${process.env.NEXT_PUBLIC_APP_URL}/ops" style="${styles.button}">Accéder au Dashboard</a>
              </div>
            </div>
          </div>
        </body>
      </html>
    `,
  })

  // Send confirmation to requester
  await transporter.sendMail({
    from: `"Swiplay" <${process.env.SMTP_USER}>`,
    to: email,
    subject: t('subject'),
    html: `
      <!DOCTYPE html>
      <html>
        <head>
          <meta charset="utf-8">
        </head>
        <body style="${styles.body}">
          <div style="${styles.container}">
            <div style="${styles.header}">
              <div style="${styles.logoContainer}">
                <span style="${styles.logoText}">SWIPLAY</span>
              </div>
            </div>
            
            <div style="${styles.content}">
              <h1 style="${styles.h1}">${t('heading')}</h1>
              <p style="${styles.text}">${t('received')}</p>
              
              <div style="text-align: center; margin: 40px 0;">
                <div style="display: inline-flex; align-items: center; justify-content: center; width: 80px; height: 80px; background: rgba(59, 130, 246, 0.1); border-radius: 50%; color: #3b82f6;">
                  <svg width="40" height="40" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <polyline points="20 6 9 17 4 12"></polyline>
                  </svg>
                </div>
              </div>
              
              <p style="${styles.text}">${t('review')}</p>
              
              <p style="${styles.text}; font-size: 14px; color: #71717a; margin-top: 30px;">
                ${t('footer')}
              </p>
            </div>
            
            <div style="${styles.footer}">
              <p>© 2025 Swiplay</p>
            </div>
          </div>
        </body>
      </html>
    `,
  })
}

export async function sendWelcomeEmail(email: string, name: string, locale: string = 'fr') {
  const t = await getTranslations({ locale, namespace: 'Emails.Welcome' })

  await transporter.sendMail({
    from: `"Swiplay" <${process.env.SMTP_USER}>`,
    to: email,
    subject: t('subject'),
    html: `
      <!DOCTYPE html>
      <html>
        <head><meta charset="utf-8"></head>
        <body style="${styles.body}">
          <div style="${styles.container}">
            <div style="${styles.header}">
              <div style="${styles.logoContainer}">
                <span style="${styles.logoText}">SWIPLAY</span>
              </div>
            </div>
            <div style="${styles.content}">
              <h1 style="${styles.h1}">${t('heading', { name })}</h1>
              <p style="${styles.text}">${t('intro')}</p>
              <div style="text-align: center;">
                <a href="${process.env.NEXT_PUBLIC_APP_URL}/creator/campaigns" style="${styles.button}">${t('cta')}</a>
              </div>
            </div>
            <div style="${styles.footer}"><p>© 2025 Swiplay</p></div>
          </div>
        </body>
      </html>
    `
  })
}

export async function sendCampaignApprovedEmail(email: string, campaignTitle: string, locale: string = 'fr') {
  const t = await getTranslations({ locale, namespace: 'Emails.CampaignApproved' })

  await transporter.sendMail({
    from: `"Swiplay" <${process.env.SMTP_USER}>`,
    to: email,
    subject: t('subject'),
    html: `
      <!DOCTYPE html>
      <html>
        <head><meta charset="utf-8"></head>
        <body style="${styles.body}">
          <div style="${styles.container}">
            <div style="${styles.header}">
              <div style="${styles.logoContainer}">
                <span style="${styles.logoText}">SWIPLAY</span>
              </div>
            </div>
            <div style="${styles.content}">
              <h1 style="${styles.h1}">${t('heading')}</h1>
              <p style="${styles.text}">${t('message', { title: campaignTitle })}</p>
              <div style="text-align: center;">
                <a href="${process.env.NEXT_PUBLIC_APP_URL}/studio/campaigns" style="${styles.button}">${t('cta')}</a>
              </div>
            </div>
            <div style="${styles.footer}"><p>© 2025 Swiplay</p></div>
          </div>
        </body>
      </html>
    `
  })
}

export async function sendNewSubmissionEmail(email: string, campaignTitle: string, creatorName: string, locale: string = 'fr') {
  const t = await getTranslations({ locale, namespace: 'Emails.NewSubmission' })

  await transporter.sendMail({
    from: `"Swiplay" <${process.env.SMTP_USER}>`,
    to: email,
    subject: t('subject'),
    html: `
      <!DOCTYPE html>
      <html>
        <head><meta charset="utf-8"></head>
        <body style="${styles.body}">
          <div style="${styles.container}">
            <div style="${styles.header}">
              <div style="${styles.logoContainer}">
                <span style="${styles.logoText}">SWIPLAY</span>
              </div>
            </div>
            <div style="${styles.content}">
              <h1 style="${styles.h1}">${t('heading')}</h1>
              <p style="${styles.text}">${t('message', { creator: creatorName, campaign: campaignTitle })}</p>
              <div style="text-align: center;">
                <a href="${process.env.NEXT_PUBLIC_APP_URL}/studio/submissions" style="${styles.button}">${t('cta')}</a>
              </div>
            </div>
            <div style="${styles.footer}"><p>© 2025 Swiplay</p></div>
          </div>
        </body>
      </html>
    `
  })
}

export async function sendSubmissionStatusEmail(email: string, campaignTitle: string, status: 'approved' | 'rejected', locale: string = 'fr') {
  const t = await getTranslations({ locale, namespace: 'Emails.SubmissionStatus' })
  const isApproved = status === 'approved'
  const color = isApproved ? '#22c55e' : '#ef4444' // Green or Red

  await transporter.sendMail({
    from: `"Swiplay" <${process.env.SMTP_USER}>`,
    to: email,
    subject: t(isApproved ? 'subject_approved' : 'subject_rejected'),
    html: `
      <!DOCTYPE html>
      <html>
        <head><meta charset="utf-8"></head>
        <body style="${styles.body}">
          <div style="${styles.container}">
            <div style="${styles.header}">
              <div style="${styles.logoContainer}">
                <span style="${styles.logoText}">SWIPLAY</span>
              </div>
            </div>
            <div style="${styles.content}" style="border-color: ${color}">
              <h1 style="${styles.h1}">${t(isApproved ? 'heading_approved' : 'heading_rejected')}</h1>
              <p style="${styles.text}">${t(isApproved ? 'message_approved' : 'message_rejected', { campaign: campaignTitle })}</p>
              <div style="text-align: center;">
                <a href="${process.env.NEXT_PUBLIC_APP_URL}/creator/submissions" style="${styles.button}">${t('cta')}</a>
              </div>
            </div>
            <div style="${styles.footer}"><p>© 2025 Swiplay</p></div>
          </div>
        </body>
      </html>
    `
  })
}

export async function sendApplicationStatusEmail(email: string, campaignTitle: string, status: 'approved' | 'rejected', creatorName: string, locale: string = 'fr') {
  const t = await getTranslations({ locale, namespace: 'Emails.ApplicationStatus' })
  const isApproved = status === 'approved'
  const color = isApproved ? '#22c55e' : '#ef4444' // Green or Red

  await transporter.sendMail({
    from: `"Swiplay" <${process.env.SMTP_USER}>`,
    to: email,
    subject: isApproved ? `Candidature acceptée ! - ${campaignTitle}` : `Mise à jour de votre candidature - ${campaignTitle}`,
    html: `
      <!DOCTYPE html>
      <html>
        <head><meta charset="utf-8"></head>
        <body style="${styles.body}">
          <div style="${styles.container}">
            <div style="${styles.header}">
              <div style="${styles.logoContainer}">
                <span style="${styles.logoText}">SWIPLAY</span>
              </div>
            </div>
            <div style="${styles.content}" style="border-color: ${color}">
              <h1 style="${styles.h1}">${isApproved ? 'Candidature Acceptée ! 🎉' : 'Candidature Refusée'}</h1>
              <p style="${styles.text}">
                ${isApproved
        ? `Félicitations <strong>${creatorName}</strong> ! Votre candidature pour la campagne <strong>${campaignTitle}</strong> a été acceptée.`
        : `Bonjour <strong>${creatorName}</strong>, votre candidature pour la campagne <strong>${campaignTitle}</strong> n'a malheureusement pas été retenue.`
      }
              </p>
              ${isApproved ? `
              <p style="${styles.text}">Vous pouvez maintenant accéder aux ressources de la campagne et soumettre votre contenu.</p>
              <div style="text-align: center;">
                <a href="${process.env.NEXT_PUBLIC_APP_URL}/creator/campaigns" style="${styles.button}">Accéder à la campagne</a>
              </div>
              ` : ''}
            </div>
            <div style="${styles.footer}"><p>© 2025 Swiplay</p></div>
          </div>
        </body>
      </html>
    `
  })
}
