/**
 * Instagram Session Hydration Script
 * Run periodically (e.g., daily via cron) to refresh cookies
 * 
 * Usage: npx ts-node lib/instagram/get-session.ts
 */

import puppeteer from 'puppeteer'
import fs from 'fs'
import path from 'path'

const SESSION_FILE = path.join(process.cwd(), 'instagram-session.json')

export interface InstagramSession {
    headers: {
        'x-csrftoken': string
        'x-ig-app-id': string
        'cookie': string
    }
    updatedAt: string
}

async function getSession(): Promise<InstagramSession | null> {
    // console.log('🔄 Starting Instagram session hydration...')

    try {
        const browser = await puppeteer.launch({
            headless: true,
            args: ['--no-sandbox', '--disable-setuid-sandbox']
        })

        const page = await browser.newPage()

        // Set realistic viewport
        await page.setViewport({ width: 1920, height: 1080 })

        // Set a real user agent
        await page.setUserAgent('Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/121.0.0.0 Safari/537.36')

        // console.log('📍 Navigating to Instagram...')
        await page.goto('https://www.instagram.com/instagram/', {
            waitUntil: 'networkidle2',
            timeout: 30000
        })

        // Wait a bit for cookies to be set
        await new Promise(resolve => setTimeout(resolve, 3000))

        const cookies = await page.cookies()

        const csrfToken = cookies.find(c => c.name === 'csrftoken')?.value || ''

        if (!csrfToken) {
            console.error('❌ No CSRF token found')
            await browser.close()
            return null
        }

        const session: InstagramSession = {
            headers: {
                'x-csrftoken': csrfToken,
                'x-ig-app-id': '936619743392459', // Instagram Web App ID
                'cookie': cookies.map(c => `${c.name}=${c.value}`).join('; ')
            },
            updatedAt: new Date().toISOString()
        }

        // Save to file
        fs.writeFileSync(SESSION_FILE, JSON.stringify(session, null, 2))
        // console.log('✅ Session saved to', SESSION_FILE)

        await browser.close()
        return session

    } catch (error) {
        console.error('❌ Error getting session:', error)
        return null
    }
}

// Load existing session from file
export function loadSession(): InstagramSession | null {
    try {
        if (fs.existsSync(SESSION_FILE)) {
            const data = fs.readFileSync(SESSION_FILE, 'utf-8')
            const session = JSON.parse(data) as InstagramSession

            // Check if session is older than 24 hours
            const updatedAt = new Date(session.updatedAt)
            const hoursSinceUpdate = (Date.now() - updatedAt.getTime()) / (1000 * 60 * 60)

            if (hoursSinceUpdate > 24) {
                // console.log('⚠️ Session is older than 24 hours, consider refreshing')
            }

            return session
        }
    } catch (error) {
        console.error('Error loading session:', error)
    }
    return null
}

// Run if called directly
if (require.main === module) {
    getSession().then(session => {
        if (session) {
            // console.log('Session ready!')
        } else {
            // console.log('Failed to get session')
            process.exit(1)
        }
    })
}

export { getSession, SESSION_FILE }
