import PDFDocument from 'pdfkit'
import fs from 'fs'
import path from 'path'

// Ensure storage directory exists
const STORAGE_DIR = path.join(process.cwd(), 'storage', 'invoices')
if (!fs.existsSync(STORAGE_DIR)) {
    fs.mkdirSync(STORAGE_DIR, { recursive: true })
}

interface StudioDetails {
    name: string
    address: string
    email: string
    siret?: string
}

interface CreatorDetails {
    name: string
    address: string
    email: string
    siret?: string
}

interface CommissionInvoiceData {
    invoiceNumber: string
    date: Date
    studio: StudioDetails
    campaignId: string // Added campaignId
    campaignTitle: string
    amount: number // Total amount paid by studio
    commissionAmount: number // 20%
}

interface MandateInvoiceData {
    invoiceNumber: string
    date: Date
    studio: StudioDetails
    creator: CreatorDetails
    campaignTitle: string
    amount: number // Amount for the creator
}

interface AccountRenderingData {
    campaignId: string
    date: Date
    studio: StudioDetails
    totalProvision: number
    totalCommission: number
    totalDisbursements: number
    creators: { name: string; amount: number }[]
}

const SWIPLAY_DETAILS = {
    name: 'SWIPLAY',
    address: '6 Rue Louise Weiss, 31200 Toulouse, France',
    email: 'contact@swiplay.com',
    siret: '987 748 365 00028',
    vat: 'TVA non applicable, art. 293 B du CGI'
}

/**
 * Generates a Commission Invoice (Swiplay -> Studio)
 */
export async function generateCommissionInvoice(data: CommissionInvoiceData): Promise<string> {
    return new Promise((resolve, reject) => {
        const doc = new PDFDocument()
        // Use campaignId in filename for deterministic linking
        const filename = `commission_${data.campaignId}.pdf`
        const filePath = path.join(STORAGE_DIR, filename)
        const stream = fs.createWriteStream(filePath)

        doc.pipe(stream)

        // Header
        doc.fontSize(20).text('FACTURE DE COMMISSION', { align: 'center' })
        doc.moveDown()

        // Swiplay Details (Seller)
        doc.fontSize(10).text(SWIPLAY_DETAILS.name)
        doc.text(SWIPLAY_DETAILS.address)
        doc.text(`SIRET: ${SWIPLAY_DETAILS.siret}`)
        doc.text(SWIPLAY_DETAILS.email)
        doc.moveDown()

        // Studio Details (Buyer)
        doc.text(`Client : ${data.studio.name}`, { align: 'right' })
        doc.text(data.studio.address, { align: 'right' })
        doc.text(data.studio.email, { align: 'right' })
        if (data.studio.siret) doc.text(`SIRET: ${data.studio.siret}`, { align: 'right' })
        doc.moveDown()

        // Invoice Info
        doc.text(`Numéro de facture : ${data.invoiceNumber}`)
        doc.text(`Date : ${data.date.toLocaleDateString('fr-FR')}`)
        doc.text(`Campagne : ${data.campaignTitle}`)
        doc.moveDown()

        // Line Items
        const y = doc.y
        doc.text('Description', 50, y)
        doc.text('Montant', 400, y, { align: 'right' })
        doc.moveTo(50, y + 15).lineTo(550, y + 15).stroke()

        doc.moveDown()
        doc.text(`Frais de service Swiplay (20%) sur campagne "${data.campaignTitle}"`, 50)
        doc.text(`${(data.commissionAmount / 100).toFixed(2)} €`, 400, doc.y - 10, { align: 'right' })

        doc.moveDown(2)
        doc.moveDown(2)
        doc.font('Helvetica-Bold').text(`Total à payer : ${(data.commissionAmount / 100).toFixed(2)} €`, { align: 'right' })
        doc.font('Helvetica')

        doc.moveDown(2)
        doc.fontSize(8).text(SWIPLAY_DETAILS.vat, { align: 'center' })

        doc.end()

        stream.on('finish', () => resolve(filePath))
        stream.on('error', reject)
    })
}

/**
 * Generates a Mandate Invoice (Creator -> Studio, issued by Swiplay)
 */
export async function generateMandateInvoice(data: MandateInvoiceData): Promise<string> {
    return new Promise((resolve, reject) => {
        const doc = new PDFDocument()
        const filename = `mandate_${data.invoiceNumber}.pdf`
        const filePath = path.join(STORAGE_DIR, filename)
        const stream = fs.createWriteStream(filePath)

        doc.pipe(stream)

        // Header
        doc.fontSize(20).text('FACTURE (MANDAT)', { align: 'center' })
        doc.moveDown()

        // Creator Details (Seller)
        doc.fontSize(10).text(`Émetteur (Créateur) : ${data.creator.name}`)
        doc.text(data.creator.address)
        doc.text(data.creator.email)
        if (data.creator.siret) doc.text(`SIRET: ${data.creator.siret}`)
        doc.moveDown()

        // Studio Details (Buyer)
        doc.text(`Client : ${data.studio.name}`, { align: 'right' })
        doc.text(data.studio.address, { align: 'right' })
        doc.text(data.studio.email, { align: 'right' })
        doc.moveDown()

        // Invoice Info
        doc.text(`Numéro de facture : ${data.invoiceNumber}`)
        doc.text(`Date : ${data.date.toLocaleDateString('fr-FR')}`)
        doc.text(`Campagne : ${data.campaignTitle}`)
        doc.moveDown()

        // Mandatory Legal Mention
        doc.fontSize(8).font('Helvetica-Oblique').text(
            `Facture émise par Swiplay au nom et pour le compte de ${data.creator.name} en application d'un mandat de facturation (Self-billing) conformément à l'article 242 nonies A de l'annexe II au CGI.`,
            { align: 'justify' }
        )
        doc.font('Helvetica')
        doc.moveDown()

        // Line Items
        doc.fontSize(10)
        const y = doc.y
        doc.text('Description', 50, y)
        doc.text('Montant', 400, y, { align: 'right' })
        doc.moveTo(50, y + 15).lineTo(550, y + 15).stroke()

        doc.moveDown()
        doc.text(`Prestation de création de contenu - Campagne "${data.campaignTitle}"`, 50)
        doc.text(`${(data.amount / 100).toFixed(2)} €`, 400, doc.y - 10, { align: 'right' })

        doc.moveDown(2)
        doc.moveDown(2)
        doc.font('Helvetica-Bold').text(`Total à payer : ${(data.amount / 100).toFixed(2)} €`, { align: 'right' })
        doc.font('Helvetica')

        doc.moveDown(2)
        doc.fontSize(8).text('TVA non applicable, art. 293 B du CGI (si micro-entrepreneur) ou Autoliquidation (si applicable).', { align: 'center' })

        doc.end()

        stream.on('finish', () => resolve(filePath))
        stream.on('error', reject)
    })
}

/**
 * Generates an Account Rendering (Reddition de Comptes)
 */
export async function generateAccountRendering(data: AccountRenderingData): Promise<string> {
    return new Promise((resolve, reject) => {
        const doc = new PDFDocument()
        const filename = `reddition_${data.campaignId}.pdf`
        const filePath = path.join(STORAGE_DIR, filename)
        const stream = fs.createWriteStream(filePath)

        doc.pipe(stream)

        // Header
        doc.fontSize(20).text('REDDITION DE COMPTES', { align: 'center' })
        doc.moveDown()

        // Swiplay Details
        doc.fontSize(10).text(SWIPLAY_DETAILS.name)
        doc.moveDown()

        // Studio Details
        doc.text(`Pour le compte de : ${data.studio.name}`)
        doc.text(`Campagne ID : ${data.campaignId}`)
        doc.text(`Date : ${data.date.toLocaleDateString('fr-FR')}`)
        doc.moveDown()

        // Summary Table
        const startY = doc.y
        doc.font('Helvetica-Bold')
        doc.text('Poste', 50, startY)
        doc.text('Montant', 400, startY, { align: 'right' })
        doc.font('Helvetica')
        doc.moveTo(50, startY + 15).lineTo(550, startY + 15).stroke()

        let currentY = startY + 30

        // Provision
        doc.text('Provision reçue (Total)', 50, currentY)
        doc.text(`${(data.totalProvision / 100).toFixed(2)} €`, 400, currentY, { align: 'right' })
        currentY += 20

        // Commission
        doc.text('Commissions Swiplay (20%)', 50, currentY)
        doc.text(`- ${(data.totalCommission / 100).toFixed(2)} €`, 400, currentY, { align: 'right' })
        currentY += 20

        // Disbursements (Creators)
        doc.text('Débours (Payés aux créateurs)', 50, currentY)
        doc.text(`- ${(data.totalDisbursements / 100).toFixed(2)} €`, 400, currentY, { align: 'right' })
        currentY += 20

        doc.moveTo(50, currentY).lineTo(550, currentY).stroke()
        currentY += 10

        // Balance
        const balance = data.totalProvision - data.totalCommission - data.totalDisbursements
        doc.font('Helvetica-Bold')
        doc.text('Solde restant', 50, currentY)
        doc.text(`${(balance / 100).toFixed(2)} €`, 400, currentY, { align: 'right' })
        doc.font('Helvetica')

        doc.moveDown(3)

        // Creators Detail
        doc.fontSize(12).text('Détail des versements aux créateurs (Débours)', { underline: true })
        doc.moveDown()
        doc.fontSize(10)

        data.creators.forEach(creator => {
            doc.text(`- ${creator.name} : ${(creator.amount / 100).toFixed(2)} €`)
        })

        doc.moveDown(2)
        doc.fontSize(8).text("Document valant reddition de comptes conformément au mandat de gestion (Art. 1993 du Code Civil).", { align: 'center' })

        doc.end()

        stream.on('finish', () => resolve(filePath))
        stream.on('error', reject)
    })
}
