import { getVideoStats } from './social-scraper'
import { getInstagramViews } from './instagram/fetch-stats'

/**
 * Fetches the view count for a video URL from the given platform.
 * Uses web scraping to extract view counts from public video pages.
 * 
 * For Instagram: Uses GraphQL API with session hydration (requires running get-session.ts first)
 * For TikTok/YouTube: Uses HTML scraping via social-scraper.ts
 * 
 * @param url - The full URL of the video
 * @param platform - The platform (tiktok, youtube, instagram)
 * @returns The view count or null if unable to fetch
 */
export async function getPlatformViews(url: string, platform: string): Promise<number | null> {
    try {
        // Validate URL
        if (!url || typeof url !== 'string') {
            console.error('[getPlatformViews] Invalid URL provided')
            return null
        }

        // Instagram uses GraphQL API with session
        if (platform === 'instagram') {
            // console.log('[getPlatformViews] Using Instagram GraphQL scraper...')
            const views = await getInstagramViews(url)

            if (views !== null && views > 0) {
                // console.log(`[getPlatformViews] Instagram: ${views} views`)
                return views
            }

            // Fallback to basic scraper if GraphQL fails
            // console.log('[getPlatformViews] Instagram GraphQL failed, trying basic scraper...')
        }

        // Use the basic HTML scraper for TikTok, YouTube, or as fallback
        const stats = await getVideoStats(url)

        if (stats && typeof stats.views === 'number' && stats.views > 0) {
            // console.log(`[getPlatformViews] Successfully fetched ${stats.views} views for ${platform} video`)
            return stats.views
        }

        // console.log(`[getPlatformViews] No views found for ${url}`)
        return null

    } catch (error) {
        console.error(`[getPlatformViews] Error fetching views for ${platform} video ${url}:`, error)
        return null
    }
}
