import { db } from '@/lib/db'
import { rateLimits } from '@/lib/db/schema'
import { eq, sql } from 'drizzle-orm'

export class DbRateLimiter {
    private interval: number

    constructor(intervalMs: number) {
        this.interval = intervalMs
    }

    async check(limit: number, key: string): Promise<boolean> {
        const now = Date.now()
        const expiresAt = now + this.interval

        // Clean up expired entries (optional, can be done via cron or lazily)
        // For now, we'll do a lazy check/reset on the specific key

        const existing = await db.query.rateLimits.findFirst({
            where: eq(rateLimits.key, key)
        })

        if (!existing) {
            await db.insert(rateLimits).values({
                key,
                count: 1,
                expiresAt,
            })
            return true
        }

        if (existing.expiresAt < now) {
            // Expired, reset
            await db.update(rateLimits)
                .set({ count: 1, expiresAt })
                .where(eq(rateLimits.key, key))
            return true
        }

        if (existing.count >= limit) {
            return false
        }

        // Increment
        await db.update(rateLimits)
            .set({ count: existing.count + 1 })
            .where(eq(rateLimits.key, key))

        return true
    }
}

// Global instances
export const campaignRateLimiter = new DbRateLimiter(60 * 60 * 1000) // 1 hour
export const authRateLimiter = new DbRateLimiter(15 * 60 * 1000) // 15 minutes
export const apiRateLimiter = new DbRateLimiter(60 * 1000) // 1 minute
