type RateLimitConfig = {
  interval: number // in milliseconds
  uniqueTokenPerInterval: number // max number of unique tokens per interval
}

export class RateLimiter {
  private tokenCache: Map<string, number[]>
  private config: RateLimitConfig

  constructor(config: RateLimitConfig) {
    this.config = config
    this.tokenCache = new Map()
  }

  check(limit: number, token: string): boolean {
    const now = Date.now()
    const windowStart = now - this.config.interval

    const timestampHistory = this.tokenCache.get(token) || []
    const validTimestamps = timestampHistory.filter((timestamp) => timestamp > windowStart)

    if (validTimestamps.length >= limit) {
      return false
    }

    validTimestamps.push(now)
    this.tokenCache.set(token, validTimestamps)

    // Cleanup
    if (this.tokenCache.size > this.config.uniqueTokenPerInterval) {
      // Simple cleanup: remove oldest entry (not perfect LRU but sufficient for simple in-memory)
      const firstKey = this.tokenCache.keys().next().value
      if (firstKey) this.tokenCache.delete(firstKey)
    }

    return true
  }
}

// Global instance for the app
export const rateLimiter = new RateLimiter({
  interval: 60 * 1000, // 1 minute
  uniqueTokenPerInterval: 500,
})