import { db } from "@/lib/db"
import { users } from "@/lib/db/schema" // import users schema if needed, though upload-avatar uses it directly? 
// actually storage service acts as S3 wrapper.
import { S3Client, PutObjectCommand } from "@aws-sdk/client-s3"
import { randomUUID } from "crypto"
import path from "path"
import fs from "fs/promises"

const MAX_STORAGE_BYTES = 5 * 1024 * 1024 * 1024 // 5GB

// S3 Client Configuration
const s3Client = new S3Client({
    region: process.env.S3_REGION || "us-east-1",
    endpoint: process.env.S3_ENDPOINT || "http://localhost:9000",
    credentials: {
        accessKeyId: process.env.S3_ACCESS_KEY_ID || "admin",
        secretAccessKey: process.env.S3_SECRET_ACCESS_KEY || "swiplay_minio_secret",
    },
    forcePathStyle: true, // Needed for Minio
})

const BUCKET_NAME = process.env.S3_BUCKET_NAME || "swiplay-assets"
const PUBLIC_URL_BASE = process.env.S3_PUBLIC_URL || "http://localhost:9000/swiplay-assets"

export async function saveFile(
    file: File,
    subDir: string = "uploads"
): Promise<{ path: string; size: number; error?: string }> {
    try {
        const fileSize = file.size
        const buffer = Buffer.from(await file.arrayBuffer())
        const ext = path.extname(file.name) || '.jpg'
        const fileName = `${randomUUID()}${ext}`
        const key = `${subDir}/${fileName}`

        let fileUrl = ""

        if (process.env.S3_ENDPOINT) {
            await s3Client.send(new PutObjectCommand({
                Bucket: BUCKET_NAME,
                Key: key,
                Body: buffer,
                ContentType: file.type,
                ACL: "public-read",
            }))
            fileUrl = `${PUBLIC_URL_BASE}/${key}`
        } else {
            console.warn("S3 credentials missing, falling back to local storage")
            const UPLOAD_DIR = path.join(process.cwd(), "public", "uploads")
            const targetDir = path.join(UPLOAD_DIR, subDir)
            await fs.mkdir(targetDir, { recursive: true })
            const filePath = path.join(targetDir, fileName)
            await fs.writeFile(filePath, buffer)
            fileUrl = `/uploads/${subDir}/${fileName}`
        }

        return { path: fileUrl, size: fileSize }
    } catch (error) {
        console.error("Storage Error:", error)
        return { path: '', size: 0, error: 'Erreur interne lors de la sauvegarde S3' }
    }
}

export async function deleteFile(fileUrl: string): Promise<{ success: boolean; error?: string }> {
    try {
        if (!fileUrl) return { success: false, error: 'No file URL provided' }

        // Extract key from URL
        // URL format: http://endpoint/bucket/subDir/filename or /uploads/subDir/filename
        let key = ''
        if (fileUrl.startsWith('http')) {
            const url = new URL(fileUrl)
            // Pathname is /bucket/key or /key depending on path style
            // With forcePathStyle=true (MinIO), it's /bucket/key
            const pathParts = url.pathname.split('/')
            // parts: ['', 'bucket', 'subDir', 'filename']
            if (pathParts.length >= 3 && pathParts[1] === BUCKET_NAME) {
                key = pathParts.slice(2).join('/')
            } else {
                // Fallback or virtual hosted style
                key = url.pathname.slice(1)
            }
        } else {
            // Local path: /uploads/subDir/filename
            key = fileUrl.startsWith('/uploads/') ? fileUrl.replace('/uploads/', 'uploads/') : fileUrl
        }

        if (process.env.S3_ENDPOINT) {
            const { DeleteObjectCommand } = await import("@aws-sdk/client-s3")
            await s3Client.send(new DeleteObjectCommand({
                Bucket: BUCKET_NAME,
                Key: key,
            }))
        } else {
            // Local deletion
            const filePath = path.join(process.cwd(), "public", key)
            await fs.unlink(filePath)
        }

        return { success: true }
    } catch (error) {
        console.error("Storage Delete Error:", error)
        return { success: false, error: 'Failed to delete file' }
    }
}
