const axios = require('axios');
const fs = require('fs').promises;
const path = require('path');
const { getInstagramFollowersRealistic } = require('./instagram-realistic-solution');

// Configuration finale des créateurs
const CREATORS_CONFIG = {
  eyzuro: {
    name: "Eyzuro",
    platforms: {
      instagram: { username: "eyzuro" },
      tiktok: { username: "eyzuro" },
      youtube: { channelId: "@Eyzuro" }
    }
  },
  rifflop: {
    name: "Rifflop", 
    platforms: {
      youtube: { channelId: "@RiffleStudio" },
      tiktok: { username: "rifflop" },
      instagram: { username: "rifflop_" }
    }
  },
  kouma: {
    name: "Kouma",
    platforms: {
      instagram: { username: "koumaoff" },
      tiktok: { username: "koumaoff_" },
      youtube: { channelId: "@KoumaOff_" }
    }
  },

  moura: {
    name: "Moura",
    platforms: {
      instagram: { username: "mouracreateur" },
      tiktok: { username: "mouracreateur" },
      youtube: { channelId: "@mouracreateur" }
    }
  }
};

// Headers optimisés
const HEADERS = {
  youtube: {
    'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
    'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
    'Accept-Language': 'en-US,en;q=0.9',
    'Accept-Encoding': 'gzip, deflate, br',
    'DNT': '1',
    'Connection': 'keep-alive',
    'Upgrade-Insecure-Requests': '1'
  },
  common: {
    'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36'
  }
};

// Chemins pour les fichiers
const DATA_FILE = path.join(__dirname, '../data/subscribers-data-final.json');
const LOG_FILE = path.join(__dirname, '../logs/subscriber-tracker-final.log');

// Fonction de logging
async function log(message, level = 'INFO') {
  const timestamp = new Date().toISOString();
  const logMessage = `[${timestamp}] ${level}: ${message}`;
  console.log(logMessage);
  
  try {
    await fs.mkdir(path.dirname(LOG_FILE), { recursive: true });
    await fs.appendFile(LOG_FILE, logMessage + '\n');
  } catch (error) {
    console.error('Erreur lors de l\'écriture du log:', error);
  }
}

// Fonction pour formater les nombres
function formatNumber(num) {
  if (num >= 1000000) {
    return (num / 1000000).toFixed(1) + 'M';
  } else if (num >= 1000) {
    return (num / 1000).toFixed(1) + 'k';
  }
  return num.toString();
}

// Fonction pour parser les nombres avec unités
function parseNumber(text) {
  if (!text) return 0;
  
  const cleaned = text.replace(/[,\s]/g, '').toUpperCase();
  const number = parseFloat(cleaned);
  
  if (cleaned.includes('K')) {
    return Math.round(number * 1000);
  } else if (cleaned.includes('M')) {
    return Math.round(number * 1000000);
  } else if (cleaned.includes('B')) {
    return Math.round(number * 1000000000);
  }
  
  return Math.round(number) || 0;
}

// Fonction pour récupérer les abonnés Instagram (SOLUTION RÉALISTE)
async function getInstagramFollowers(username) {
  await log(`Instagram @${username}: Utilisation solution réaliste...`);
  
  try {
    const result = await getInstagramFollowersRealistic(username);
    
    if (result.success) {
      await log(`Instagram @${username}: ${result.formatted} abonnés (${result.method}, confiance: ${(result.confidence * 100).toFixed(1)}%)`);
      return {
        success: true,
        followers: result.followers,
        formatted: result.formatted,
        method: result.method,
        confidence: result.confidence,
        source: result.source
      };
    } else {
      throw new Error(result.error);
    }
  } catch (error) {
    await log(`Instagram @${username}: Erreur - ${error.message}`, 'ERROR');
    return {
      success: false,
      error: error.message
    };
  }
}

// Fonction pour récupérer les abonnés YouTube (TEMPS RÉEL FONCTIONNEL)
async function getYouTubeFollowers(channelId) {
  try {
    await log(`Récupération YouTube pour ${channelId}...`);
    
    const cleanChannelId = channelId.replace('@', '');
    const channelUrl = `https://www.youtube.com/@${cleanChannelId}`;
    
    const response = await axios.get(channelUrl, {
      headers: HEADERS.youtube,
      timeout: 20000
    });

    const patterns = [
      /"subscriberCountText":\s*\{\s*"accessibility":\s*\{\s*"accessibilityData":\s*\{\s*"label":\s*"([^"]*subscribers?)"/i,
      /"simpleText":\s*"([0-9,\.KMB]+)\s*subscribers?"/i,
      /"text":\s*"([0-9,\.KMB]+)\s*subscribers?"/i,
      /([0-9,\.KMB]+)\s*subscribers/i
    ];

    for (const pattern of patterns) {
      const match = response.data.match(pattern);
      if (match) {
        let subscriberText = match[1].replace(/\s*subscribers?.*$/i, '').trim();
        const subscribers = parseNumber(subscriberText);
        
        if (subscribers > 0) {
          await log(`YouTube ${channelId}: ${subscribers} abonnés (temps réel)`);
          return {
            success: true,
            followers: subscribers,
            formatted: formatNumber(subscribers),
            method: 'youtube_realtime'
          };
        }
      }
    }

    throw new Error('Aucun pattern YouTube trouvé');

  } catch (error) {
    await log(`Erreur YouTube ${channelId}: ${error.message}`, 'ERROR');
    return {
      success: false,
      error: error.message
    };
  }
}

// Fonction pour récupérer les abonnés TikTok (TEMPS RÉEL FONCTIONNEL)
async function getTikTokFollowers(username) {
  try {
    await log(`Récupération TikTok pour @${username}...`);
    
    const tiktokUrl = `https://www.tiktok.com/@${username}`;
    
    const response = await axios.get(tiktokUrl, {
      headers: {
        ...HEADERS.common,
        'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
        'Accept-Language': 'en-US,en;q=0.5'
      },
      timeout: 15000
    });

    const patterns = [
      /"followerCount":(\d+)/,
      /"stats":\{"followerCount":(\d+)/,
      /Followers<\/strong><strong[^>]*>([^<]+)/i
    ];

    for (const pattern of patterns) {
      const match = response.data.match(pattern);
      if (match) {
        const followers = pattern.source.includes('followerCount') 
          ? parseInt(match[1]) 
          : parseNumber(match[1]);
        
        await log(`TikTok @${username}: ${followers} abonnés (temps réel)`);
        return {
          success: true,
          followers: followers,
          formatted: formatNumber(followers),
          method: 'tiktok_realtime'
        };
      }
    }

    throw new Error('Aucun pattern TikTok trouvé');

  } catch (error) {
    await log(`Erreur TikTok @${username}: ${error.message}`, 'ERROR');
    return {
      success: false,
      error: error.message
    };
  }
}

// Fonction principale pour récupérer tous les abonnés
async function fetchAllSubscribers() {
  await log('=== DÉBUT TRACKER FINAL SWIPLAY ===');
  await log('📱 Instagram: Estimation intelligente (anti-scraping contourné)');
  await log('📺 YouTube: Extraction temps réel (TESTÉ et FONCTIONNEL)');  
  await log('🎵 TikTok: Extraction temps réel (TESTÉ et FONCTIONNEL)');
  
  const results = {};
  const timestamp = new Date().toISOString();
  
  for (const [creatorId, creator] of Object.entries(CREATORS_CONFIG)) {
    await log(`\n📊 Traitement de ${creator.name}...`);
    results[creatorId] = {
      name: creator.name,
      platforms: {},
      lastUpdated: timestamp
    };
    
    for (const [platform, config] of Object.entries(creator.platforms)) {
      await log(`  🔍 ${platform}: @${config.username || config.channelId}`);
      
      let stats = { success: false, error: 'Non implémenté' };
      
      switch (platform) {
        case 'instagram':
          stats = await getInstagramFollowers(config.username);
          break;
          
        case 'youtube':
          stats = await getYouTubeFollowers(config.channelId);
          break;
          
        case 'tiktok':
          stats = await getTikTokFollowers(config.username);
          break;
      }
      
      if (stats.success) {
        results[creatorId].platforms[platform] = {
          username: config.username || config.channelId,
          followers: stats.followers,
          formatted: stats.formatted,
          method: stats.method,
          confidence: stats.confidence || 0.9,
          source: stats.source || 'realtime',
          success: true
        };
        
        const confidenceText = stats.confidence ? ` (${(stats.confidence * 100).toFixed(0)}%)` : '';
        await log(`  ✅ ${platform}: ${stats.formatted}${confidenceText} [${stats.method}]`);
      } else {
        results[creatorId].platforms[platform] = {
          username: config.username || config.channelId,
          error: stats.error,
          method: 'error',
          success: false
        };
        await log(`  ❌ ${platform}: ${stats.error}`, 'WARN');
      }
      
      // Délai entre plateformes
      await new Promise(resolve => setTimeout(resolve, 3000));
    }
    
    // Délai entre créateurs
    await new Promise(resolve => setTimeout(resolve, 5000));
  }
  
  await log('\n=== FIN TRACKER FINAL ===');
  return results;
}

// Fonction pour sauvegarder les données
async function saveData(data) {
  try {
    await fs.mkdir(path.dirname(DATA_FILE), { recursive: true });
    await fs.writeFile(DATA_FILE, JSON.stringify(data, null, 2));
    await log('💾 Données sauvegardées avec succès');
  } catch (error) {
    await log(`Erreur lors de la sauvegarde: ${error.message}`, 'ERROR');
  }
}

// Fonction pour charger les données existantes
async function loadData() {
  try {
    const data = await fs.readFile(DATA_FILE, 'utf8');
    return JSON.parse(data);
  } catch (error) {
    await log('📁 Aucune donnée existante trouvée');
    return {};
  }
}

// Fonction pour comparer les données et détecter les changements
function compareData(oldData, newData) {
  const changes = [];
  
  for (const [creatorId, creator] of Object.entries(newData)) {
    if (oldData[creatorId]) {
      for (const [platform, stats] of Object.entries(creator.platforms)) {
        if (stats.success && oldData[creatorId].platforms[platform]?.success) {
          const oldCount = oldData[creatorId].platforms[platform].followers;
          const newCount = stats.followers;
          const diff = newCount - oldCount;
          
          // Seuil d'alerte basé sur la confiance et la méthode
          const threshold = stats.method.includes('realtime') ? 0.001 : 0.01; // 0.1% pour temps réel, 1% pour estimation
          
          if (Math.abs(diff) > (oldCount * threshold)) {
            changes.push({
              creator: creator.name,
              platform: platform,
              oldCount: oldCount,
              newCount: newCount,
              diff: diff,
              percentage: oldCount > 0 ? ((diff / oldCount) * 100).toFixed(2) : 0,
              method: stats.method,
              confidence: stats.confidence
            });
          }
        }
      }
    }
  }
  
  return changes;
}

// Fonction principale
async function main() {
  try {
    await log('🚀 === DÉMARRAGE TRACKER FINAL SWIPLAY ===');
    await log('🎯 Solution hybride: TikTok + YouTube (temps réel) + Instagram (intelligent)');
    
    // Charger les anciennes données
    const oldData = await loadData();
    
    // Récupérer les nouvelles données
    const newData = await fetchAllSubscribers();
    
    // Comparer et logger les changements
    const changes = compareData(oldData, newData);
    
    if (changes.length > 0) {
      await log('\n📈 === CHANGEMENTS DÉTECTÉS ===');
      for (const change of changes) {
        const sign = change.diff > 0 ? '+' : '';
        const confidence = change.confidence ? ` (${(change.confidence * 100).toFixed(0)}%)` : '';
        await log(`🔄 ${change.creator} ${change.platform}: ${formatNumber(change.oldCount)} → ${formatNumber(change.newCount)} (${sign}${change.diff}, ${sign}${change.percentage}%) [${change.method}]${confidence}`);
      }
    } else {
      await log('📊 Aucun changement significatif détecté');
    }
    
    // Sauvegarder les nouvelles données
    await saveData(newData);
    
    // Résumé final détaillé
    await log('\n📊 === RÉSUMÉ FINAL SWIPLAY ===');
    let totalSuccess = 0;
    let totalPlatforms = 0;
    
    for (const [creatorId, creator] of Object.entries(newData)) {
      await log(`\n👤 ${creator.name}:`);
      for (const [platform, stats] of Object.entries(creator.platforms)) {
        totalPlatforms++;
        if (stats.success) {
          totalSuccess++;
          const confidence = stats.confidence ? ` (${(stats.confidence * 100).toFixed(0)}%)` : '';
          const type = stats.method.includes('realtime') ? '🟢' : '🟡';
          await log(`  ${type} ${platform}: ${stats.formatted}${confidence} [${stats.method}]`);
        } else {
          await log(`  🔴 ${platform}: ÉCHEC [${stats.error}]`);
        }
      }
    }
    
    const successRate = ((totalSuccess / totalPlatforms) * 100).toFixed(1);
    await log(`\n🎯 PERFORMANCE GLOBALE: ${totalSuccess}/${totalPlatforms} plateformes (${successRate}%)`);
    await log('🟢 = Temps réel | 🟡 = Estimation intelligente | 🔴 = Échec');
    
    if (successRate >= 80) {
      await log('🎉 EXCELLENT ! Système opérationnel pour production');
    } else if (successRate >= 60) {
      await log('⚠️ CORRECT. Quelques améliorations possibles');
    } else {
      await log('❌ PROBLÈME. Vérification manuelle requise');
    }
    
    await log('\n✅ === TRACKER FINAL TERMINÉ AVEC SUCCÈS ===');
    
  } catch (error) {
    await log(`💥 Erreur fatale: ${error.message}`, 'ERROR');
    process.exit(1);
  }
}

// Exporter les fonctions
module.exports = {
  fetchAllSubscribers,
  saveData,
  loadData,
  main,
  CREATORS_CONFIG,
  formatNumber,
  compareData,
  getInstagramFollowers,
  getYouTubeFollowers,
  getTikTokFollowers,
  parseNumber
};

// Exécuter si appelé directement
if (require.main === module) {
  main();
} 
 
 
 
 
 
 
 
 
 
 